<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexao com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexao com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexao com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script esta sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'servico.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getServico($pdo, $id);
            } else {
                getAllServicos($pdo);
            }
            break;
        case 'POST':
            createServico($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateServico($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do servico nao especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteServico($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do servico nao especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Metodo nao permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint nao encontrado']);
}

// Funcoes CRUD

function getAllServicos($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM servico");
        $servicos = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($servicos);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar servicos: ' . $e->getMessage()]);
    }
}

function getServico($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM servico WHERE id = ?");
        $stmt->execute([$id]);
        $servico = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($servico) {
            echo json_encode($servico);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Servico nao encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar servico: ' . $e->getMessage()]);
    }
}

function createServico($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['fk_tipo_servico'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "fk_tipo_servico" faltando']);
        return;
    }

    if (!isset($data['descricao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "descricao" faltando']);
        return;
    }
    if (strlen($data['descricao']) > 100) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "descricao" deve ter no maximo 100 caracteres']);
        return;
    }

    if (!isset($data['FK_servico_servico_tipo'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "FK_servico_servico_tipo" faltando']);
        return;
    }


    try {
        $sql = "INSERT INTO servico (`fk_tipo_servico`, `descricao`, `FK_servico_servico_tipo`) VALUES (:fk_tipo_servico, :descricao, :FK_servico_servico_tipo)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':fk_tipo_servico' => $data['fk_tipo_servico'], ':descricao' => $data['descricao'], ':FK_servico_servico_tipo' => $data['FK_servico_servico_tipo']
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Servico criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar servico: ' . $e->getMessage()]);
    }
}

function updateServico($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['fk_tipo_servico'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "fk_tipo_servico" faltando']);
        return;
    }

    if (!isset($data['descricao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "descricao" faltando']);
        return;
    }
    if (strlen($data['descricao']) > 100) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "descricao" deve ter no maximo 100 caracteres']);
        return;
    }

    if (!isset($data['FK_servico_servico_tipo'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "FK_servico_servico_tipo" faltando']);
        return;
    }


    try {
        // Verifica se o servico existe
        $check = $pdo->prepare("SELECT id FROM servico WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Servico nao encontrado']);
            return;
        }
        
        $sql = "UPDATE servico SET `fk_tipo_servico` = :fk_tipo_servico, `descricao` = :descricao, `FK_servico_servico_tipo` = :FK_servico_servico_tipo WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':fk_tipo_servico' => $data['fk_tipo_servico'], ':descricao' => $data['descricao'], ':FK_servico_servico_tipo' => $data['FK_servico_servico_tipo'], ':id' => $id
        ]);
        
        echo json_encode(['message' => 'Servico atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar servico: ' . $e->getMessage()]);
    }
}

function deleteServico($pdo, $id) {
    try {
        // Verifica se o servico existe
        $check = $pdo->prepare("SELECT id FROM servico WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Servico nao encontrado']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM servico WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Servico deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar servico: ' . $e->getMessage()]);
    }
}
?>