<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexão com o .env - caminho relativo ajustado
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexão com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexão com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento simplificado
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script está sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'produtos.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getProduto($pdo, $id);
            } else {
                getAllProdutos($pdo);
            }
            break;
        case 'POST':
            createProduto($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateProduto($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do produto não especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteProduto($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do produto não especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint não encontrado']);
}

// Funções para manipulação dos produtos

function getAllProdutos($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM produto");
        $produtos = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($produtos);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar produtos: ' . $e->getMessage()]);
    }
}

function getProduto($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM produto WHERE id = ?");
        $stmt->execute([$id]);
        $produto = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($produto) {
            echo json_encode($produto);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Produto não encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar produto: ' . $e->getMessage()]);
    }
}

function createProduto($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados inválidos']);
        return;
    }
    
    // Validação dos campos obrigatórios
    $requiredFields = ['tipo', 'un', 'ncm', 'produto', 'custo', 'fabricante', 'modelo', 'status'];
    foreach ($requiredFields as $field) {
        if (!isset($data[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Campo obrigatório faltando: $field"]);
            return;
        }
    }
    
    try {
        $sql = "INSERT INTO produto (
            foto, tipo, un, ncm, produto, descricao, armazem, local, localizacao, 
            qtd_min, qtd_max, for_pad, custo, venda, serie, fabricante, modelo, status
        ) VALUES (
            :foto, :tipo, :un, :ncm, :produto, :descricao, :armazem, :local, :localizacao, 
            :qtd_min, :qtd_max, :for_pad, :custo, :venda, :serie, :fabricante, :modelo, :status
        )";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':foto' => $data['foto'] ?? null,
            ':tipo' => $data['tipo'],
            ':un' => $data['un'],
            ':ncm' => $data['ncm'],
            ':produto' => $data['produto'],
            ':descricao' => $data['descricao'] ?? null,
            ':armazem' => $data['armazem'] ?? null,
            ':local' => $data['local'] ?? null,
            ':localizacao' => $data['localizacao'] ?? null,
            ':qtd_min' => $data['qtd_min'] ?? null,
            ':qtd_max' => $data['qtd_max'] ?? null,
            ':for_pad' => $data['for_pad'] ?? null,
            ':custo' => $data['custo'],
            ':venda' => $data['venda'] ?? null,
            ':serie' => $data['serie'] ?? null,
            ':fabricante' => $data['fabricante'],
            ':modelo' => $data['modelo'],
            ':status' => $data['status']
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Produto criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar produto: ' . $e->getMessage()]);
    }
}

function updateProduto($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados inválidos']);
        return;
    }
    
    try {
        // Verifica se o produto existe
        $check = $pdo->prepare("SELECT id FROM produto WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Produto não encontrado']);
            return;
        }
        
        $sql = "UPDATE produto SET 
            foto = :foto, 
            tipo = :tipo, 
            un = :un, 
            ncm = :ncm, 
            produto = :produto, 
            descricao = :descricao, 
            armazem = :armazem, 
            local = :local, 
            localizacao = :localizacao, 
            qtd_min = :qtd_min, 
            qtd_max = :qtd_max, 
            for_pad = :for_pad, 
            custo = :custo, 
            venda = :venda, 
            serie = :serie, 
            fabricante = :fabricante, 
            modelo = :modelo, 
            status = :status 
        WHERE id = :id";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id' => $id,
            ':foto' => $data['foto'] ?? null,
            ':tipo' => $data['tipo'] ?? null,
            ':un' => $data['un'] ?? null,
            ':ncm' => $data['ncm'] ?? null,
            ':produto' => $data['produto'] ?? null,
            ':descricao' => $data['descricao'] ?? null,
            ':armazem' => $data['armazem'] ?? null,
            ':local' => $data['local'] ?? null,
            ':localizacao' => $data['localizacao'] ?? null,
            ':qtd_min' => $data['qtd_min'] ?? null,
            ':qtd_max' => $data['qtd_max'] ?? null,
            ':for_pad' => $data['for_pad'] ?? null,
            ':custo' => $data['custo'] ?? null,
            ':venda' => $data['venda'] ?? null,
            ':serie' => $data['serie'] ?? null,
            ':fabricante' => $data['fabricante'] ?? null,
            ':modelo' => $data['modelo'] ?? null,
            ':status' => $data['status'] ?? null
        ]);
        
        echo json_encode(['message' => 'Produto atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar produto: ' . $e->getMessage()]);
    }
}

function deleteProduto($pdo, $id) {
    try {
        // Verifica se o produto existe
        $check = $pdo->prepare("SELECT id FROM produto WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Produto não encontrado']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM produto WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Produto deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar produto: ' . $e->getMessage()]);
    }
}
?>