<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexão com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexão com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexão com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script está sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'eqp_modelos.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getModelo($pdo, $id);
            } else {
                getAllModelos($pdo);
            }
            break;
        case 'POST':
            createModelo($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateModelo($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do modelo não especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteModelo($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do modelo não especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint não encontrado']);
}

// Funções CRUD

function getAllModelos($pdo) {
    try {
        $stmt = $pdo->query("
            SELECT m.*, 
                   f.nome as fabricante_nome,
                   t.nome as tipo_nome,
                   tam.nome as tamanho_nome
            FROM eq_modelos m
            LEFT JOIN eq_fabricantes f ON m.id_fabricante = f.id
            LEFT JOIN eq_tipo t ON m.id_tipo = t.id
            LEFT JOIN eq_tamanho tam ON m.id_tamanho = tam.id
        ");
        $modelos = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($modelos);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar modelos: ' . $e->getMessage()]);
    }
}

function getModelo($pdo, $id) {
    try {
        $stmt = $pdo->prepare("
            SELECT m.*, 
                   f.nome as fabricante_nome,
                   t.nome as tipo_nome,
                   tam.nome as tamanho_nome
            FROM eq_modelos m
            LEFT JOIN eq_fabricantes f ON m.id_fabricante = f.id
            LEFT JOIN eq_tipo t ON m.id_tipo = t.id
            LEFT JOIN eq_tamanho tam ON m.id_tamanho = tam.id
            WHERE m.id = ?
        ");
        $stmt->execute([$id]);
        $modelo = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($modelo) {
            echo json_encode($modelo);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Modelo não encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar modelo: ' . $e->getMessage()]);
    }
}

function createModelo($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    $requiredFields = ['id_fabricante', 'id_tipo', 'id_tamanho', 'nome', 'foto'];
    foreach ($requiredFields as $field) {
        if (!isset($data[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Campo obrigatório faltando: $field"]);
            return;
        }
    }
    
    // Validação do tamanho máximo do nome
    if (strlen($data['nome']) > 200) {
        http_response_code(400);
        echo json_encode(['error' => 'O nome do modelo deve ter no máximo 200 caracteres']);
        return;
    }
    
    try {
        $sql = "INSERT INTO eq_modelos 
                (id_fabricante, id_tipo, id_tamanho, nome, foto) 
                VALUES 
                (:id_fabricante, :id_tipo, :id_tamanho, :nome, :foto)";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id_fabricante' => $data['id_fabricante'],
            ':id_tipo' => $data['id_tipo'],
            ':id_tamanho' => $data['id_tamanho'],
            ':nome' => $data['nome'],
            ':foto' => $data['foto']
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Modelo criado com sucesso']);
    } catch (PDOException $e) {
        // Verifica se é erro de duplicação (nome único)
        if ($e->getCode() == 23000) {
            http_response_code(400);
            echo json_encode(['error' => 'Já existe um modelo com este nome']);
        } else {
            http_response_code(500);
            echo json_encode(['error' => 'Erro ao criar modelo: ' . $e->getMessage()]);
        }
    }
}

function updateModelo($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados inválidos']);
        return;
    }
    
    // Validação do tamanho máximo do nome
    if (isset($data['nome']) && strlen($data['nome']) > 200) {
        http_response_code(400);
        echo json_encode(['error' => 'O nome do modelo deve ter no máximo 200 caracteres']);
        return;
    }
    
    try {
        // Verifica se o modelo existe
        $check = $pdo->prepare("SELECT id FROM eq_modelos WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Modelo não encontrado']);
            return;
        }
        
        // Monta a query dinamicamente com os campos fornecidos
        $fields = [];
        $params = [':id' => $id];
        
        if (isset($data['id_fabricante'])) $fields[] = 'id_fabricante = :id_fabricante';
        if (isset($data['id_tipo'])) $fields[] = 'id_tipo = :id_tipo';
        if (isset($data['id_tamanho'])) $fields[] = 'id_tamanho = :id_tamanho';
        if (isset($data['nome'])) $fields[] = 'nome = :nome';
        if (isset($data['foto'])) $fields[] = 'foto = :foto';
        
        if (empty($fields)) {
            http_response_code(400);
            echo json_encode(['error' => 'Nenhum campo válido para atualização']);
            return;
        }
        
        $sql = "UPDATE eq_modelos SET " . implode(', ', $fields) . " WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        
        // Adiciona os parâmetros
        if (isset($data['id_fabricante'])) $params[':id_fabricante'] = $data['id_fabricante'];
        if (isset($data['id_tipo'])) $params[':id_tipo'] = $data['id_tipo'];
        if (isset($data['id_tamanho'])) $params[':id_tamanho'] = $data['id_tamanho'];
        if (isset($data['nome'])) $params[':nome'] = $data['nome'];
        if (isset($data['foto'])) $params[':foto'] = $data['foto'];
        
        $stmt->execute($params);
        
        echo json_encode(['message' => 'Modelo atualizado com sucesso']);
    } catch (PDOException $e) {
        // Verifica se é erro de duplicação (nome único)
        if ($e->getCode() == 23000) {
            http_response_code(400);
            echo json_encode(['error' => 'Já existe um modelo com este nome']);
        } else {
            http_response_code(500);
            echo json_encode(['error' => 'Erro ao atualizar modelo: ' . $e->getMessage()]);
        }
    }
}

function deleteModelo($pdo, $id) {
    try {
        // Verifica se o modelo existe
        $check = $pdo->prepare("SELECT id FROM eq_modelos WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Modelo não encontrado']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM eq_modelos WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Modelo deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar modelo: ' . $e->getMessage()]);
    }
}
?>