<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexão com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexão com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexão com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script está sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'contrato_maquina.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getMaquina($pdo, $id);
            } else {
                $fk_contrato = isset($_GET['fk_contrato']) ? (int)$_GET['fk_contrato'] : null;
                $situacao = isset($_GET['situacao']) ? $_GET['situacao'] : null;
                getAllMaquinas($pdo, $fk_contrato, $situacao);
            }
            break;
        case 'POST':
            createMaquina($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateMaquina($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID da máquina não especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteMaquina($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID da máquina não especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint não encontrado']);
}

// Funções CRUD

function getAllMaquinas($pdo, $fk_contrato = null, $situacao = null) {
    try {
        $sql = "SELECT * FROM contrato_maquina";
        $conditions = [];
        $params = [];
        
        if ($fk_contrato) {
            $conditions[] = "fk_contrato = ?";
            $params[] = $fk_contrato;
        }
        
        if ($situacao) {
            $conditions[] = "situacao = ?";
            $params[] = $situacao;
        }
        
        if (!empty($conditions)) {
            $sql .= " WHERE " . implode(" AND ", $conditions);
        }
        
        $sql .= " ORDER BY equipamento ASC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        $maquinas = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($maquinas);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar máquinas: ' . $e->getMessage()]);
    }
}

function getMaquina($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM contrato_maquina WHERE id = ?");
        $stmt->execute([$id]);
        $maquina = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($maquina) {
            echo json_encode($maquina);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Máquina não encontrada']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar máquina: ' . $e->getMessage()]);
    }
}

function createMaquina($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    // Campos obrigatórios
    $requiredFields = [
        'fk_contrato', 'fk_matriz', 'fk_nome_matriz', 'modelo', 
        'equipamento', 'serie', 'endereco', 'bairro', 'cidade', 
        'uf', 'setor_area', 'situacao'
    ];
    
    foreach ($requiredFields as $field) {
        if (!isset($data[$field])) {  
            http_response_code(400);
            echo json_encode(['error' => "Campo '$field' é obrigatório"]);
            return;
        }
    }
    
    // Validações de tamanho
    $validations = [
        'fk_nome_matriz' => 50,
        'site_filial' => 50,
        'modelo' => 30,
        'equipamento' => 50,
        'serie' => 50,
        'endereco' => 50,
        'numero' => 50,
        'bairro' => 50,
        'cidade' => 50,
        'uf' => 2,
        'setor_area' => 50,
        'ip_number' => 50,
        'googlemaps_point' => 100
    ];
    
    foreach ($validations as $field => $maxLength) {
         if (!isset($data[$field])) {
            if (strlen($data[$field]) > $maxLength) {
                http_response_code(400);
                echo json_encode(['error' => "O campo '$field' deve ter no máximo $maxLength caracteres"]);
                return;
            }
        }
    }
    
    // Validação do ENUM
    $situacaoValues = ['ATIVO', 'INATIVO', 'AGUARDANDO'];
    if (!in_array($data['situacao'], $situacaoValues)) {
        http_response_code(400);
        echo json_encode(['error' => 'Situação inválida. Valores permitidos: ' . implode(', ', $situacaoValues)]);
        return;
    }
    
    try {
        $sql = "INSERT INTO contrato_maquina (
            fk_contrato, fk_matriz, fk_nome_matriz, site_filial, modelo, 
            equipamento, serie, endereco, numero, bairro, cidade, uf, 
            setor_area, ip_number, situacao, obs, googlemaps_point
        ) VALUES (
            :fk_contrato, :fk_matriz, :fk_nome_matriz, :site_filial, :modelo, 
            :equipamento, :serie, :endereco, :numero, :bairro, :cidade, :uf, 
            :setor_area, :ip_number, :situacao, :obs, :googlemaps_point
        )";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':fk_contrato' => $data['fk_contrato'],
            ':fk_matriz' => $data['fk_matriz'],
            ':fk_nome_matriz' => $data['fk_nome_matriz'],
            ':site_filial' => $data['site_filial'] ?? null,
            ':modelo' => $data['modelo'],
            ':equipamento' => $data['equipamento'],
            ':serie' => $data['serie'],
            ':endereco' => $data['endereco'],
            ':numero' => $data['numero'] ?? null,
            ':bairro' => $data['bairro'],
            ':cidade' => $data['cidade'],
            ':uf' => $data['uf'],
            ':setor_area' => $data['setor_area'],
            ':ip_number' => $data['ip_number'] ?? null,
            ':situacao' => $data['situacao'],
            ':obs' => $data['obs'] ?? null,
            ':googlemaps_point' => $data['googlemaps_point'] ?? null
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Máquina criada com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar máquina: ' . $e->getMessage()]);
    }
}

function updateMaquina($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    // Campos obrigatórios
    $requiredFields = [
        'fk_contrato', 'fk_matriz', 'fk_nome_matriz', 'modelo', 
        'equipamento', 'serie', 'endereco', 'bairro', 'cidade', 
        'uf', 'setor_area', 'situacao'
    ];
    
    foreach ($requiredFields as $field) {
        if (!isset($data[$field])) {
            http_response_code(400);
            echo json_encode(['error' => "Campo '$field' é obrigatório"]);
            return;
        }
    }
    
    // Validações de tamanho (mesmo do create)
    $validations = [
        'fk_nome_matriz' => 50,
        'site_filial' => 50,
        'modelo' => 30,
        'equipamento' => 50,
        'serie' => 50,
        'endereco' => 50,
        'numero' => 50,
        'bairro' => 50,
        'cidade' => 50,
        'uf' => 2,
        'setor_area' => 50,
        'ip_number' => 50,
        'googlemaps_point' => 100
    ];
    
    foreach ($validations as $field => $maxLength) {
        if (isset($data[$field])) {
            if (strlen($data[$field]) > $maxLength) {
                http_response_code(400);
                echo json_encode(['error' => "O campo '$field' deve ter no máximo $maxLength caracteres"]);
                return;
            }
        }
    }
    
    // Validação do ENUM
    $situacaoValues = ['ATIVO', 'INATIVO', 'AGUARDANDO'];
    if (!in_array($data['situacao'], $situacaoValues)) {
        http_response_code(400);
        echo json_encode(['error' => 'Situação inválida. Valores permitidos: ' . implode(', ', $situacaoValues)]);
        return;
    }
    
    try {
        // Verifica se a máquina existe
        $check = $pdo->prepare("SELECT id FROM contrato_maquina WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Máquina não encontrada']);
            return;
        }
        
        $sql = "UPDATE contrato_maquina SET 
                fk_contrato = :fk_contrato,
                fk_matriz = :fk_matriz,
                fk_nome_matriz = :fk_nome_matriz,
                site_filial = :site_filial,
                modelo = :modelo,
                equipamento = :equipamento,
                serie = :serie,
                endereco = :endereco,
                numero = :numero,
                bairro = :bairro,
                cidade = :cidade,
                uf = :uf,
                setor_area = :setor_area,
                ip_number = :ip_number,
                situacao = :situacao,
                obs = :obs,
                googlemaps_point = :googlemaps_point
                WHERE id = :id";
                
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id' => $id,
            ':fk_contrato' => $data['fk_contrato'],
            ':fk_matriz' => $data['fk_matriz'],
            ':fk_nome_matriz' => $data['fk_nome_matriz'],
            ':site_filial' => $data['site_filial'] ?? null,
            ':modelo' => $data['modelo'],
            ':equipamento' => $data['equipamento'],
            ':serie' => $data['serie'],
            ':endereco' => $data['endereco'],
            ':numero' => $data['numero'] ?? null,
            ':bairro' => $data['bairro'],
            ':cidade' => $data['cidade'],
            ':uf' => $data['uf'],
            ':setor_area' => $data['setor_area'],
            ':ip_number' => $data['ip_number'] ?? null,
            ':situacao' => $data['situacao'],
            ':obs' => $data['obs'] ?? null,
            ':googlemaps_point' => $data['googlemaps_point'] ?? null
        ]);
        
        echo json_encode(['message' => 'Máquina atualizada com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar máquina: ' . $e->getMessage()]);
    }
}

function deleteMaquina($pdo, $id) {
    try {
        // Verifica se a máquina existe
        $check = $pdo->prepare("SELECT id FROM contrato_maquina WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Máquina não encontrada']);
            return;
        }
        
        // Verifica se existem leituras associadas a esta máquina
        $checkLeituras = $pdo->prepare("SELECT id FROM contrato_maquina_leituras WHERE fk_contrato_makina = ?");
        $checkLeituras->execute([$id]);
        
        if ($checkLeituras->fetch()) {
            http_response_code(400);
            echo json_encode(['error' => 'Não é possível deletar a máquina pois existem leituras associadas a ela']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM contrato_maquina WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Máquina deletada com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar máquina: ' . $e->getMessage()]);
    }
}
?>