<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexao com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexao com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexao com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script esta sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'area.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getArea($pdo, $id);
            } else {
                getAllAreas($pdo);
            }
            break;
        case 'POST':
            createArea($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateArea($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do area nao especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteArea($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do area nao especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Metodo nao permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint nao encontrado']);
}

// Funcoes CRUD

function getAllAreas($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM area");
        $areas = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($areas);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar areas: ' . $e->getMessage()]);
    }
}

function getArea($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM area WHERE id = ?");
        $stmt->execute([$id]);
        $area = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($area) {
            echo json_encode($area);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Area nao encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar area: ' . $e->getMessage()]);
    }
}

function createArea($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['descricao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "descricao" faltando']);
        return;
    }
    if (strlen($data['descricao']) > 30) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "descricao" deve ter no maximo 30 caracteres']);
        return;
    }


    try {
        $sql = "INSERT INTO area (`descricao`) VALUES (:descricao)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':descricao' => $data['descricao']
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Area criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar area: ' . $e->getMessage()]);
    }
}

function updateArea($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['descricao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "descricao" faltando']);
        return;
    }
    if (strlen($data['descricao']) > 30) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "descricao" deve ter no maximo 30 caracteres']);
        return;
    }


    try {
        // Verifica se o area existe
        $check = $pdo->prepare("SELECT id FROM area WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Area nao encontrado']);
            return;
        }
        
        $sql = "UPDATE area SET `descricao` = :descricao WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':descricao' => $data['descricao'], ':id' => $id
        ]);
        
        echo json_encode(['message' => 'Area atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar area: ' . $e->getMessage()]);
    }
}

function deleteArea($pdo, $id) {
    try {
        // Verifica se o area existe
        $check = $pdo->prepare("SELECT id FROM area WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Area nao encontrado']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM area WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Area deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar area: ' . $e->getMessage()]);
    }
}
?>