<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexao com o .env
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexao com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexao com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script esta sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'solucao.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getSolucao($pdo, $id);
            } else {
                getAllSolucaos($pdo);
            }
            break;
        case 'POST':
            createSolucao($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateSolucao($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do solucao nao especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteSolucao($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do solucao nao especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Metodo nao permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint nao encontrado']);
}

// Funcoes CRUD

function getAllSolucaos($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM solucao");
        $solucaos = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($solucaos);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar solucaos: ' . $e->getMessage()]);
    }
}

function getSolucao($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM solucao WHERE id = ?");
        $stmt->execute([$id]);
        $solucao = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($solucao) {
            echo json_encode($solucao);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Solucao nao encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar solucao: ' . $e->getMessage()]);
    }
}

function createSolucao($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['fk_servico'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "fk_servico" faltando']);
        return;
    }

    if (!isset($data['solucao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "solucao" faltando']);
        return;
    }
    if (strlen($data['solucao']) > 255) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "solucao" deve ter no maximo 255 caracteres']);
        return;
    }

    if (!isset($data['FK_solucao_servico'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "FK_solucao_servico" faltando']);
        return;
    }


    try {
        $sql = "INSERT INTO solucao (`fk_servico`, `solucao`, `FK_solucao_servico`) VALUES (:fk_servico, :solucao, :FK_solucao_servico)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':fk_servico' => $data['fk_servico'], ':solucao' => $data['solucao'], ':FK_solucao_servico' => $data['FK_solucao_servico']
        ]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Solucao criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar solucao: ' . $e->getMessage()]);
    }
}

function updateSolucao($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados invalidos']);
        return;
    }
    
    if (!isset($data['fk_servico'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "fk_servico" faltando']);
        return;
    }

    if (!isset($data['solucao'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "solucao" faltando']);
        return;
    }
    if (strlen($data['solucao']) > 255) {
        http_response_code(400);
        echo json_encode(['error' => 'O campo "solucao" deve ter no maximo 255 caracteres']);
        return;
    }

    if (!isset($data['FK_solucao_servico'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Campo "FK_solucao_servico" faltando']);
        return;
    }


    try {
        // Verifica se o solucao existe
        $check = $pdo->prepare("SELECT id FROM solucao WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Solucao nao encontrado']);
            return;
        }
        
        $sql = "UPDATE solucao SET `fk_servico` = :fk_servico, `solucao` = :solucao, `FK_solucao_servico` = :FK_solucao_servico WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':fk_servico' => $data['fk_servico'], ':solucao' => $data['solucao'], ':FK_solucao_servico' => $data['FK_solucao_servico'], ':id' => $id
        ]);
        
        echo json_encode(['message' => 'Solucao atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar solucao: ' . $e->getMessage()]);
    }
}

function deleteSolucao($pdo, $id) {
    try {
        // Verifica se o solucao existe
        $check = $pdo->prepare("SELECT id FROM solucao WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Solucao nao encontrado']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM solucao WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Solucao deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar solucao: ' . $e->getMessage()]);
    }
}
?>