<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE");
header("Access-Control-Allow-Headers: Content-Type");

// Conexão com o .env - caminho relativo ajustado
$dotenv = parse_ini_file(__DIR__ . '../../.env');
$dbHost = $dotenv['DB_HOST'];
$dbUser = $dotenv['DB_USER'];
$dbPass = $dotenv['DB_PASSWORD'];
$dbName = $dotenv['DB_NAME'];

// Conexão com o banco de dados
try {
    $pdo = new PDO("mysql:host=$dbHost;dbname=$dbName;charset=utf8", $dbUser, $dbPass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Falha na conexão com o banco de dados: ' . $e->getMessage()]);
    exit;
}

// Roteamento simplificado
$method = $_SERVER['REQUEST_METHOD'];
$id = isset($_GET['id']) ? (int)$_GET['id'] : null;

// Verifica se o script está sendo acessado diretamente
if (basename($_SERVER['SCRIPT_FILENAME']) === 'eqp_tipos.php') {
    switch ($method) {
        case 'GET':
            if ($id) {
                getTipo($pdo, $id);
            } else {
                getAllTipos($pdo);
            }
            break;
        case 'POST':
            createTipo($pdo);
            break;
        case 'PUT':
            if ($id) {
                updateTipo($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do tipo não especificado']);
            }
            break;
        case 'DELETE':
            if ($id) {
                deleteTipo($pdo, $id);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'ID do tipo não especificado']);
            }
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Método não permitido']);
    }
} else {
    http_response_code(404);
    echo json_encode(['error' => 'Endpoint não encontrado']);
}

// Funções para manipulação dos tipos

function getAllTipos($pdo) {
    try {
        $stmt = $pdo->query("SELECT * FROM eq_tipo");
        $tipos = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo json_encode($tipos);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar tipos: ' . $e->getMessage()]);
    }
}

function getTipo($pdo, $id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM eq_tipo WHERE id = ?");
        $stmt->execute([$id]);
        $tipo = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($tipo) {
            echo json_encode($tipo);
        } else {
            http_response_code(404);
            echo json_encode(['error' => 'Tipo não encontrado']);
        }
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao buscar tipo: ' . $e->getMessage()]);
    }
}

function createTipo($pdo) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || !isset($data['nome'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados inválidos ou campo "nome" faltando']);
        return;
    }
    
    try {
        $sql = "INSERT INTO eq_tipo (nome) VALUES (:nome)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([':nome' => $data['nome']]);
        
        $id = $pdo->lastInsertId();
        http_response_code(201);
        echo json_encode(['id' => $id, 'message' => 'Tipo criado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao criar tipo: ' . $e->getMessage()]);
    }
}

function updateTipo($pdo, $id) {
    $data = json_decode(file_get_contents('php://input'), true);
    
    if (!$data || !isset($data['nome'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Dados inválidos ou campo "nome" faltando']);
        return;
    }
    
    try {
        // Verifica se o tipo existe
        $check = $pdo->prepare("SELECT id FROM eq_tipo WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Tipo não encontrado']);
            return;
        }
        
        $sql = "UPDATE eq_tipo SET nome = :nome WHERE id = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':id' => $id,
            ':nome' => $data['nome']
        ]);
        
        echo json_encode(['message' => 'Tipo atualizado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao atualizar tipo: ' . $e->getMessage()]);
    }
}

function deleteTipo($pdo, $id) {
    try {
        // Verifica se o tipo existe
        $check = $pdo->prepare("SELECT id FROM eq_tipo WHERE id = ?");
        $check->execute([$id]);
        
        if (!$check->fetch()) {
            http_response_code(404);
            echo json_encode(['error' => 'Tipo não encontrado']);
            return;
        }
        
        $stmt = $pdo->prepare("DELETE FROM eq_tipo WHERE id = ?");
        $stmt->execute([$id]);
        
        echo json_encode(['message' => 'Tipo deletado com sucesso']);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Erro ao deletar tipo: ' . $e->getMessage()]);
    }
}